<?php

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Group_Control_Image_Size;
use Elementor\Repeater;
use Makbet\Elementor\Makbet_Base_Widgets;


class Makbet_Elementor_Testimonials extends Makbet_Base_Widgets {

    /**
     * Get widget name.
     *
     * Retrieve testimonial widget name.
     *
     * @return string Widget name.
     * @since  1.0.0
     * @access public
     *
     */
    public function get_name() {
        return 'makbet-testimonials';
    }

    /**
     * Get widget title.
     *
     * Retrieve testimonial widget title.
     *
     * @return string Widget title.
     * @since  1.0.0
     * @access public
     *
     */
    public function get_title() {
        return esc_html__('Makbet Testimonials', 'makbet');
    }

    /**
     * Get widget icon.
     *
     * Retrieve testimonial widget icon.
     *
     * @return string Widget icon.
     * @since  1.0.0
     * @access public
     *
     */
    public function get_icon() {
        return 'eicon-testimonial';
    }

    public function get_script_depends() {
        return ['makbet-elementor-testimonial'];
    }

    public function get_style_depends() {
        return [ 'e-swiper' ];
    }

    public function get_categories() {
        return array('makbet-addons');
    }

    /**
     * Register testimonial widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since  1.0.0
     * @access protected
     */
    protected function register_controls() {
        $this->start_controls_section(
            'section_testimonial',
            [
                'label' => esc_html__('Testimonials', 'makbet'),
            ]
        );

        $repeater = new Repeater();
        $repeater->add_control(
            'testimonial_icon',
            [
                'label' => esc_html__('Icon', 'makbet'),
                'type'  => Controls_Manager::ICONS,
            ]
        );

        $repeater->add_control(
            'testimonial_rating',
            [
                'label'   => esc_html__('Rating', 'makbet'),
                'default' => 5,
                'type'    => Controls_Manager::SELECT,
                'options' => [
                    0 => esc_html__('Hidden', 'makbet'),
                    1 => esc_html__('Very poor', 'makbet'),
                    2 => esc_html__('Not that bad', 'makbet'),
                    3 => esc_html__('Average', 'makbet'),
                    4 => esc_html__('Good', 'makbet'),
                    5 => esc_html__('Perfect', 'makbet'),
                ]
            ]
        );
        $repeater->add_control(
            'testimonial_content',
            [
                'label'       => esc_html__('Content', 'makbet'),
                'type'        => Controls_Manager::TEXTAREA,
                'default'     => 'Click edit button to change this text. Lorem ipsum dolor sit amet, consectetur adipiscing elit. Ut elit tellus, luctus nec ullamcorper mattis, pulvinar dapibus leo.',
                'label_block' => true,
                'rows'        => '6',
            ]
        );

        $repeater->add_control(
            'testimonial_name',
            [
                'label'   => esc_html__('Name', 'makbet'),
                'default' => 'John Doe',
                'type'    => Controls_Manager::TEXT,
            ]
        );

        $repeater->add_control(
            'testimonial_job',
            [
                'label'   => esc_html__('Job', 'makbet'),
                'default' => 'Design',
                'type'    => Controls_Manager::TEXT,
            ]
        );

        $repeater->add_control(
            'testimonial_link',
            [
                'label'       => esc_html__('Link to', 'makbet'),
                'placeholder' => esc_html__('https://your-link.com', 'makbet'),
                'type'        => Controls_Manager::URL,
            ]
        );

        $repeater->add_control(
            'testimonial_image',
            [
                'label'      => esc_html__('Choose Image', 'makbet'),
                'type'       => Controls_Manager::MEDIA,
                'show_label' => false,
            ]
        );


        $this->add_control(
            'testimonials',
            [
                'label'       => esc_html__('Items', 'makbet'),
                'type'        => Controls_Manager::REPEATER,
                'fields'      => $repeater->get_controls(),
                'title_field' => '{{{ testimonial_name }}}',
            ]
        );
        $this->add_control(
            'icon-view',
            [
                'label'        => esc_html__('Icon-view', 'makbet'),
                'type'         => Controls_Manager::SELECT,
                'options'      => [
                    'default' => esc_html__('Default', 'makbet'),
                    'stacked' => esc_html__('Stacked', 'makbet'),
                    'framed'  => esc_html__('Framed', 'makbet'),
                ],
                'default'      => 'default',
                'prefix_class' => 'elementor-view-',
            ]
        );
        $this->add_control(
            'shape',
            [
                'label'        => esc_html__('Shape', 'makbet'),
                'type'         => Controls_Manager::SELECT,
                'options'      => [
                    'circle' => esc_html__('Circle', 'makbet'),
                    'square' => esc_html__('Square', 'makbet'),
                ],
                'default'      => 'circle',
                'condition'    => [
                    'icon-view!' => 'default',
                ],
                'prefix_class' => 'elementor-shape-',
            ]
        );
        $this->add_group_control(
            Elementor\Group_Control_Image_Size::get_type(),
            [
                'name'      => 'testimonial_image',
                'default'   => 'full',
                'separator' => 'none',
            ]
        );

        $this->add_control(
            'testimonial_layout',
            [
                'label'   => esc_html__('Layout', 'makbet'),
                'type'    => Controls_Manager::SELECT,
                'default' => '1',
                'options' => [
                    '1' => 'Style 1',
                    '2' => 'Style 2',
                    '3' => 'Style 3',
                ],
            ]
        );



        $this->add_control(
            'view',
            [
                'label'   => esc_html__('View', 'makbet'),
                'type'    => Controls_Manager::HIDDEN,
                'default' => 'traditional',
            ]
        );

        $this->end_controls_section();

        // WRAPPER STYLE
        $this->start_controls_section(
            'section_style_testimonial',
            [
                'label' => esc_html__('Wrapper', 'makbet'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'padding_testimonial',
            [
                'label'      => esc_html__('Padding', 'makbet'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .inner' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .layout-3 .elementor-testimonial-item' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'condition' => [
                    'testimonial_layout!' => '3'
                ],
            ]
        );

        $this->add_responsive_control(
            'layout_3_gap',
            [
                'label'      => esc_html__('Gap', 'makbet'),
                'type'       => Controls_Manager::SLIDER,
                'range'      => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .elementor-testimonial-item-wrapper.layout-wrapper-3' => 'Gap: {{SIZE}}{{UNIT}}',
                ],
                'condition' => [
                    'testimonial_layout' => '3'
                ],
            ]
        );
        $this->add_responsive_control(
            'padding_testimonial_layout_3',
            [
                'label'      => esc_html__('Padding', 'makbet'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .layout-3 .elementor-testimonial-item' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'condition' => [
                    'testimonial_layout' => '3'
                ],
            ]
        );

        $this->add_responsive_control(
            'margin_testimonial',
            [
                'label'      => esc_html__('Margin', 'makbet'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em'],
                'selectors'  => [
                    '{{WRAPPER}} .inner' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'color_testimonial_background',
            [
                'label'     => esc_html__('Background Color', 'makbet'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .inner' => 'background: {{VALUE}};',
                    '{{WRAPPER}} .layout-3 .elementor-testimonial-item' => 'background: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name'     => 'testimonial_box-shadow',
                'selector' => '{{WRAPPER}} .inner',
            ]
        );


        $this->add_responsive_control(
            'testimonial_alignment',
            [
                'label'       => esc_html__('Alignment Content', 'makbet'),
                'type'        => Controls_Manager::CHOOSE,
                'options'     => [
                    'left'   => [
                        'title' => esc_html__('Left', 'makbet'),
                        'icon'  => 'eicon-text-align-left',
                    ],
                    'center' => [
                        'title' => esc_html__('Center', 'makbet'),
                        'icon'  => 'eicon-text-align-center',
                    ],
                    'right'  => [
                        'title' => esc_html__('Right', 'makbet'),
                        'icon'  => 'eicon-text-align-right',
                    ],
                ],
                'label_block' => false,
                'selectors'   => [
                    '{{WRAPPER}} .elementor-testimonial-item-wrapper .inner'                                  => 'text-align: {{VALUE}};',
                    '{{WRAPPER}} .elementor-testimonial-item-wrapper .content'                                => 'text-align: {{VALUE}};',
                    '{{WRAPPER}} .elementor-testimonial-item-wrapper .testimonial-content-text'               => 'justify-content: {{VALUE}}',
                    '{{WRAPPER}} .elementor-testimonial-item-wrapper .layout-1 .name'                         => 'justify-content: {{VALUE}}',
                    '{{WRAPPER}} .elementor-testimonial-item-wrapper .layout-2 .name'                         => 'justify-content: {{VALUE}}',
                ],
            ]
        );
        $this->add_responsive_control(
            'align_caption',
            [
                'label'     => esc_html__('Alignment Info ', 'makbet'),
                'type'      => Controls_Manager::CHOOSE,
                'options'   => [
                    'flex-start' => [
                        'title' => esc_html__('Left', 'makbet'),
                        'icon'  => 'eicon-h-align-left',
                    ],
                    'center'     => [
                        'title' => esc_html__('Center', 'makbet'),
                        'icon'  => 'eicon-h-align-center',
                    ],
                    'flex-end'   => [
                        'title' => esc_html__('Right', 'makbet'),
                        'icon'  => 'eicon-h-align-right',
                    ],
                ],
                'default'   => 'flex-start',
                'selectors' => [
                    '{{WRAPPER}} .elementor-testimonial-item-wrapper .testimonial-caption'              => 'justify-content: {{VALUE}}',
                    '{{WRAPPER}} .elementor-testimonial-item-wrapper .elementor-testimonial-rating'     => 'justify-content: {{VALUE}}',
                    '{{WRAPPER}} .elementor-testimonial-item-wrapper .testimonial-caption '                             => 'justify-content: {{VALUE}}',
                    '{{WRAPPER}} .elementor-testimonial-item-wrapper .testimonial-caption .caption-top' => 'justify-content: {{VALUE}}',
                    '{{WRAPPER}} .elementor-testimonial-item-wrapper .layout-1 .icon'                   => 'justify-content: {{VALUE}}',
                    '{{WRAPPER}} .elementor-testimonial-item-wrapper .layout-2 .icon'                   => 'justify-content: {{VALUE}}',
                    '{{WRAPPER}} .elementor-testimonial-item-wrapper .layout-3 .testimonial-caption'                   => 'justify-content: {{VALUE}}',
                    '{{WRAPPER}} .elementor-testimonial-item-wrapper.layout-wrapper-3 .layout-3 .testimonial-image'                   => 'justify-content: {{VALUE}}'
                ],
            ]
        );
        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name'        => 'wrapper_border',
                'placeholder' => '1px',
                'default'     => '1px',
                'selector'    => '{{WRAPPER}} .inner',
                'separator'   => 'before',
            ]
        );

        $this->add_control(
            'border_radius_box',
            [
                'label'      => esc_html__('Border Radius', 'makbet'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .inner' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'section_style_testimonial_img',
            [
                'label' => esc_html__('Image', 'makbet'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'image_gap',
            [
                'label'      => esc_html__('Gap', 'makbet'),
                'type'       => Controls_Manager::SLIDER,
                'range'      => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .testimonial-caption .caption-top' => 'Gap: {{SIZE}}{{UNIT}}',
                    '{{WRAPPER}} .layout-3 .elementor-testimonial-item' => 'Gap: {{SIZE}}{{UNIT}}',
                ],
            ]
        );
        $this->add_responsive_control(
            'testimonial_width_img',
            [
                'label'      => esc_html__('Width', 'makbet'),
                'type'       => Controls_Manager::SLIDER,
                'range'      => [
                    'px' => [
                        'min' => 10,
                        'max' => 500,
                    ],
                    'vw' => [
                        'min' => 1,
                        'max' => 100,
                    ],
                ],
                'size_units' => ['px', '%', 'vw'],
                'selectors'  => [
                    '{{WRAPPER}} .elementor-testimonial-image img'           => 'width: {{SIZE}}{{UNIT}}',
                    '{{WRAPPER}} .test-image'           => 'width: {{SIZE}}{{UNIT}}',
                    '{{WRAPPER}} .layout-wrapper-3 .layout-3 .elementor-testimonial-image'           => 'width: {{SIZE}}{{UNIT}}',
                ],
            ]
        );

        $this->add_responsive_control(
            'testimonial_height_img',
            [
                'label'      => esc_html__('Height', 'makbet'),
                'type'       => Controls_Manager::SLIDER,
                'range'      => [
                    'px' => [
                        'min' => 10,
                        'max' => 500,
                    ],
                    'vh' => [
                        'min' => 10,
                        'max' => 100,
                    ],
                ],
                'size_units' => ['px', 'vh', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .elementor-testimonial-image img' => 'height: {{SIZE}}{{UNIT}}',
                    '{{WRAPPER}} .test-image' => 'height: {{SIZE}}{{UNIT}}',
                    '{{WRAPPER}} .layout-wrapper-3 .layout-3 .elementor-testimonial-image'           => 'height: {{SIZE}}{{UNIT}}',

                ],
            ]
        );

        $this->add_control(
            'testimonial_radius_img',
            [
                'label'      => esc_html__('Border Radius', 'makbet'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .elementor-testimonial-image img' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .test-image' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'padding_testimonial_img',
            [
                'label'      => esc_html__('Padding', 'makbet'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .elementor-testimonial-image' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .test-image' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'margin_testimonial_img',
            [
                'label'      => esc_html__('Margin', 'makbet'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em'],
                'selectors'  => [
                    '{{WRAPPER}} .elementor-testimonial-image' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .test-image' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Content style
        $this->start_controls_section(
            'section_style_testimonial_style',
            [
                'label' => esc_html__('Content', 'makbet'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'content_content_color',
            [
                'label'     => esc_html__('Color', 'makbet'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .content' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name'     => 'content_typography',
                'selector' => '{{WRAPPER}} .content',
            ]
        );
        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name'        => 'wrapper_content_border',
                'placeholder' => '1px',
                'default'     => '1px',
                'selector'    => '{{WRAPPER}} .content',
                'separator'   => 'before',
            ]
        );
        $this->add_control(
            'border_radius_content',
            [
                'label'      => esc_html__('Border Radius', 'makbet'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .testimonial-content-text' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        $this->add_responsive_control(
            'content_spacing',
            [
                'label'      => esc_html__('Margin', 'makbet'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .content' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        $this->add_responsive_control(
            'content_padding',
            [
                'label'      => esc_html__('Padding', 'makbet'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        $this->end_controls_section();


        // Name.
        $this->start_controls_section(
            'section_style_testimonial_name',
            [
                'label' => esc_html__('Name', 'makbet'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'name_text_color',
            [
                'label'     => esc_html__('Color', 'makbet'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .name, {{WRAPPER}} .name a'                                 => 'color: {{VALUE}};',
                    '{{WRAPPER}} .caption-bottom .name, {{WRAPPER}} .caption-bottom .name a' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .elementor-testimonial-item-wrapper .testimonial-style-3 .testimonial-info-wrapper .testimonial-info .testimonial-name' => 'color: {{VALUE}};',


                ],
            ]
        );

        $this->add_control(
            'name_text_color_hover',
            [
                'label'     => esc_html__('Color Hover', 'makbet'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .inner .name:hover, {{WRAPPER}} .inner .name a:hover'                   => 'color: {{VALUE}};',
                    '{{WRAPPER}} .caption-bottom .name:hover, {{WRAPPER}} .caption-bottom .name a:hover' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .elementor-testimonial-item-wrapper .testimonial-style-3 .testimonial-info-wrapper .testimonial-info .testimonial-name:hover' => 'color: {{VALUE}};',

                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name'     => 'name_typography',
                'selector' => '{{WRAPPER}} .name, {{WRAPPER}} .elementor-testimonial-item-wrapper .testimonial-style-3 .testimonial-info-wrapper .testimonial-info .testimonial-name',
            ]
        );

        $this->add_responsive_control(
            'name_margin',
            [
                'label'      => esc_html__('Margin', 'makbet'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .name' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',

                ],
            ]
        );

        $this->end_controls_section();

        // Job.
        $this->start_controls_section(
            'section_style_testimonial_job',
            [
                'label' => esc_html__('Job', 'makbet'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'job_text_color',
            [
                'label'     => esc_html__('Color', 'makbet'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .job'                 => 'color: {{VALUE}};',
                    '{{WRAPPER}} .caption-bottom .job' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .elementor-testimonial-item-wrapper .testimonial-style-3 .testimonial-info-wrapper .testimonial-info .testimonial-job' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'job_text_color_hover',
            [
                'label'     => esc_html__('Color Hover', 'makbet'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .inner:hover .job' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .elementor-testimonial-item-wrapper .testimonial-style-3 .testimonial-info-wrapper .testimonial-info .testimonial-job:hover' => 'color: {{VALUE}};',
                ],
            ]
        );


        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name'     => 'job_typography',
                'selector' => '{{WRAPPER}} .job, {{WRAPPER}} .elementor-testimonial-item-wrapper .testimonial-style-3 .testimonial-info-wrapper .testimonial-info .testimonial-job',

            ]
        );

        $this->end_controls_section();

        // rating
        $this->start_controls_section(
            'section_style_testimonial_rating',
            [
                'label' => esc_html__('Rating', 'makbet'),
                'tab'   => Controls_Manager::TAB_STYLE,
                'condition' => [
                    'testimonial_layout' => '2'
                ],
            ]
        );

        $this->add_control(
            'rating_color',
            [
                'label'     => esc_html__('Color', 'makbet'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .elementor-testimonial-rating' => 'color: {{VALUE}};',
                ],
            ]
        );
        $this->add_control(
            'rating_icon_color',
            [
                'label'     => esc_html__('Icon Color', 'makbet'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .elementor-testimonial-rating i' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'rating_spacing',
            [
                'label'      => esc_html__('Margin', 'makbet'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .elementor-testimonial-rating' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Icon
        $this->start_controls_section(
            'section_style_testimonial_icon',
            [
                'label' => esc_html__('Icon', 'makbet'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'icon_color',
            [
                'label'     => esc_html__('Color', 'makbet'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .icon i' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'icon_color_hover',
            [
                'label'     => esc_html__('Color Hover', 'makbet'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .inner:hover .icon i' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'icon_background',
            [
                'label'     => esc_html__('Background', 'makbet'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .icon' => 'background-color: {{VALUE}} !important;',
                ],
            ]
        );

        $this->add_responsive_control(
            'icon_size_inner',
            [
                'label'      => esc_html__('Size', 'makbet'),
                'type'       => Controls_Manager::SLIDER,
                'range'      => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .inner .icon' => 'font-size: {{SIZE}}{{UNIT}} !important;',
                ],
            ]
        );

        $this->add_responsive_control(
            'icon_top_spacing',
            [
                'label'      => esc_html__('Margin Icon', 'makbet'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .inner .icon' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        $this->add_responsive_control(
            'icon_padding',
            [
                'label'      => esc_html__('Padding Icon', 'makbet'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}}.elementor-view-stacked .icon i' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}}.elementor-view-framed .icon i'  => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .icon i'                        => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();
        // Carousel column
        $this->get_controls_column();
        // Carousel options
        $this->get_control_carousel();
    }

    /**
     * Render testimonial widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since  1.0.0
     * @access protected
     */
    protected function render() {
        $settings = $this->get_settings_for_display();
        if (!empty($settings['testimonials']) && is_array($settings['testimonials'])) {
            $this->add_render_attribute('wrapper', 'class', 'layout-wrapper-'. esc_attr($settings['testimonial_layout']) . ' elementor-testimonial-item-wrapper');
            // Row
            $this->get_data_elementor_columns();
            $this->add_render_attribute('inner', 'class', 'layout-' . esc_attr($settings['testimonial_layout']));
            // Container
            $this->add_render_attribute('container', 'class', ' elementor-testimonials-swiper');

            $this->add_render_attribute('container', 'data-count', count($settings['testimonials']));
            // Item
            $this->add_render_attribute('item', 'class', 'elementor-grid-item elementor-testimonial-item');
            $this->add_render_attribute('details', 'class', 'details');

            ?>
            <div <?php $this->print_render_attribute_string('wrapper'); ?>>
                <?php if ($settings['testimonial_layout'] == '3'): ?>
                    <div class="elementor-custom-swiper-pagination swiper-container testimonial-style-3">
                        <div class="elementor-inner-custom-swiper-pagination swiper-wrapper">
                            <?php foreach ($settings['testimonials'] as $testimonial): ?>
                                <div class="swiper-custom swiper-slide elementor-testimonial-item">
                                    <div class="swiper-slide testimonial-info-wrapper">
                                        <div class="testimonial-image">
                                            <?php $this->render_image($settings, $testimonial); ?>
                                            <?php if (!empty($testimonial['testimonial_icon']['value'])): ?>
                                                <div class="icon"><?php \Elementor\Icons_Manager::render_icon($testimonial['testimonial_icon'], ['aria-hidden' => 'true']); ?></div>
                                            <?php endif; ?>
                                        </div>
                                        <div class="testimonial-info">
                                            <?php
                                            $testimonial_name_html = $testimonial['testimonial_name'];
                                            if (!empty($testimonial['testimonial_link']['url'])) {
                                                $testimonial_name_html = '<a href="' . esc_url($testimonial['testimonial_link']['url']) . '">' . esc_html($testimonial_name_html) . '</a>';
                                            }
                                            echo '<span class="testimonial-name">' . $testimonial_name_html . '</span>';
                                            ?>
                                            <?php if (!empty($testimonial['testimonial_job'])): ?>
                                                <span class="testimonial-job"><?php echo esc_html($testimonial['testimonial_job']); ?></span>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                <?php endif; ?>

                <div <?php $this->print_render_attribute_string('container'); ?>>
                    <div <?php $this->print_render_attribute_string('inner'); ?>>
                        <?php foreach ($settings['testimonials'] as $testimonial): ?>
                            <div <?php $this->print_render_attribute_string('item'); ?>>
                                <?php if ($settings['testimonial_layout'] == '1'): ?>
                                    <div class="item-inner inner">
                                        <div class="testimonial-content-text">
                                            <?php if (!empty($testimonial['testimonial_icon']['value'])): ?>
                                                <div class="icon"><?php \Elementor\Icons_Manager::render_icon($testimonial['testimonial_icon'], ['aria-hidden' => 'true']); ?></div>
                                            <?php endif; ?>

                                            <?php if (!empty($testimonial['testimonial_content'])) : ?>
                                                <div class="content"><?php echo sprintf('%s', $testimonial['testimonial_content']); ?></div>
                                            <?php endif; ?>

                                            <div class="testimonial-caption">
                                                <div class="caption-top">
                                                    <?php $this->render_image($settings, $testimonial); ?>
                                                    <div <?php $this->print_render_attribute_string('details'); ?>>
                                                        <?php
                                                        $testimonial_name_html = $testimonial['testimonial_name'];
                                                        if (!empty($testimonial['testimonial_link']['url'])) :
                                                            $testimonial_name_html = '<a href="' . esc_url($testimonial['testimonial_link']['url']) . '">' . esc_html($testimonial_name_html) . '</a>';
                                                        endif;
                                                        printf('<span class="name">%s</span>', $testimonial_name_html);
                                                        ?>
                                                        <?php if ($testimonial['testimonial_job']): ?>
                                                            <span class="job"><?php echo esc_html($testimonial['testimonial_job']); ?></span>
                                                        <?php endif; ?>
                                                    </div>
                                                </div>
                                            </div>

                                            <div class="testimonial-rating">
                                                <?php $this->render_rating($testimonial); ?>
                                            </div>
                                        </div>
                                    </div>
                                <?php endif; ?>
                                <?php if ($settings['testimonial_layout'] == '2'): ?>
                                    <div class="item-inner inner">
                                        <div class="testimonial-content-text">
                                            <?php if (!empty($testimonial['testimonial_icon']['value'])): ?>
                                                <div class="icon"><?php \Elementor\Icons_Manager::render_icon($testimonial['testimonial_icon'], ['aria-hidden' => 'true']); ?></div>
                                            <?php endif; ?>

                                            <div class="testimonial-rating">
                                                <?php $this->render_rating($testimonial); ?>
                                            </div>

                                            <?php if (!empty($testimonial['testimonial_content'])) : ?>
                                                <div class="content"><?php echo sprintf('%s', $testimonial['testimonial_content']); ?></div>
                                            <?php endif; ?>

                                            <div class="testimonial-caption">
                                                <div class="caption-top">
                                                    <?php $this->render_image($settings, $testimonial); ?>
                                                    <div <?php $this->print_render_attribute_string('details'); ?>>
                                                        <?php
                                                        $testimonial_name_html = $testimonial['testimonial_name'];
                                                        if (!empty($testimonial['testimonial_link']['url'])) :
                                                            $testimonial_name_html = '<a href="' . esc_url($testimonial['testimonial_link']['url']) . '">' . esc_html($testimonial_name_html) . '</a>';
                                                        endif;
                                                        printf('<span class="name">%s</span>', $testimonial_name_html);
                                                        ?>
                                                        <?php if ($testimonial['testimonial_job']): ?>
                                                            <span class="job"><?php echo esc_html($testimonial['testimonial_job']); ?></span>
                                                        <?php endif; ?>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                <?php endif; ?>
                                <?php if ($settings['testimonial_layout'] == '3'): ?>
                                    <div class="testimonial-image">
                                        <?php $this->render_image($settings, $testimonial); ?>

                                    </div>
                                    <div class="item-inner inner">
                                        <div class="testimonial-content-text">
                                            <div class="testimonial-rating">
                                                <?php $this->render_rating($testimonial); ?>
                                            </div>
                                            <?php if (!empty($testimonial['testimonial_content'])) : ?>
                                                <div class="content"><?php echo sprintf('%s', $testimonial['testimonial_content']); ?></div>
                                            <?php endif; ?>
                                        </div>
                                        <div class="testimonial-caption">
                                            <div class="caption-top">
                                                <div <?php $this->print_render_attribute_string('details'); ?>>
                                                    <?php
                                                    $testimonial_name_html = $testimonial['testimonial_name'];
                                                    if (!empty($testimonial['testimonial_link']['url'])) :
                                                        $testimonial_name_html = '<a href="' . esc_url($testimonial['testimonial_link']['url']) . '">' . esc_html($testimonial_name_html) . '</a>';
                                                    endif;
                                                    printf('<span class="name">%s</span>', $testimonial_name_html);
                                                    ?>
                                                    <?php if ($testimonial['testimonial_job']): ?>
                                                        <span class="job"><?php echo esc_html($testimonial['testimonial_job']); ?></span>
                                                    <?php endif; ?>
                                                </div>
                                            </div>
                                        </div>
                                    </div>

                                <?php endif; ?>
                            </div>
                        <?php endforeach; ?>

                    </div>
                    <?php $this->get_swiper_navigation(count($settings['testimonials'])); ?>
                </div>

            </div>
            <?php
        }
    }

    private function render_image($settings, $testimonial) {
        if (!empty($testimonial['testimonial_image']['url'])) :
            ?>
            <div class="elementor-testimonial-image">
                <?php
                $testimonial['testimonial_image_size']             = $settings['testimonial_image_size'];
                $testimonial['testimonial_image_custom_dimension'] = $settings['testimonial_image_custom_dimension'];
                echo Group_Control_Image_Size::get_attachment_image_html($testimonial, 'testimonial_image');
                ?>
            </div>
        <?php
        endif;
    }

    private function render_rating($testimonial) {
        if ($testimonial['testimonial_rating'] && $testimonial['testimonial_rating'] > 0) {
            echo '<div class="elementor-testimonial-rating">';
            for ($i = 1; $i <= 5; $i++) {
                if ($i <= $testimonial['testimonial_rating']) {
                    echo '<i class="makbet-icon-rating   active" aria-hidden="true"></i>';
                }
            }
            echo '</div>';
        }
    }
}

$widgets_manager->register(new Makbet_Elementor_Testimonials());
