<?php

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

use Makbet\Elementor\Makbet_Base_Widgets;
use Elementor\Controls_Manager;
use Elementor\Core\Kits\Documents\Tabs\Global_Colors;
use Elementor\Core\Kits\Documents\Tabs\Global_Typography;
use Elementor\Group_Control_Background;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Border;

/**
 * Class Makbet_Elementor_Blog
 */
class Makbet_Elementor_Portfolio extends Makbet_Base_Widgets {

    public function get_name() {
        return 'makbet-portfolio';
    }

    public function get_title() {
        return esc_html__('Makbet Portfolio', 'makbet');
    }

    /**
     * Get widget icon.
     *
     * Retrieve testimonial widget icon.
     *
     * @return string Widget icon.
     * @since  1.0.0
     * @access public
     *
     */
    public function get_icon() {
        return 'eicon-archive';
    }

    public function get_categories() {
        return array('makbet-addons');
    }

    public function get_script_depends() {
        return ['makbet-elementor-portfolio'];
    }

    public function get_style_depends() {
        return ['e-swiper'];
    }

    protected function register_controls() {
        $this->start_controls_section(
            'section_query',
            [
                'label' => esc_html__('Query', 'makbet'),
                'tab'   => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'portfolio-style',
            [
                'label'   => esc_html__('Style', 'makbet'),
                'type'    => Controls_Manager::SELECT,
                'default' => '1',
                'options' => [
                    1 => esc_html__('Style 1', 'makbet'),
                    2 => esc_html__('Style 2', 'makbet'),
                    3 => esc_html__('Style 3', 'makbet'),
                ],
            ]
        );

        $this->add_control(
            'posts_per_page',
            [
                'label'   => esc_html__('Posts Per Page', 'makbet'),
                'type'    => Controls_Manager::NUMBER,
                'default' => 6,
            ]
        );

        $this->add_control(
            'orderby',
            [
                'label'   => esc_html__('Order By', 'makbet'),
                'type'    => Controls_Manager::SELECT,
                'default' => 'post_date',
                'options' => [
                    'post_date'  => esc_html__('Date', 'makbet'),
                    'post_title' => esc_html__('Title', 'makbet'),
                    'menu_order' => esc_html__('Menu Order', 'makbet'),
                    'rand'       => esc_html__('Random', 'makbet'),
                ],
            ]
        );

        $this->add_control(
            'order',
            [
                'label'   => esc_html__('Order', 'makbet'),
                'type'    => Controls_Manager::SELECT,
                'default' => 'desc',
                'options' => [
                    'asc'  => esc_html__('ASC', 'makbet'),
                    'desc' => esc_html__('DESC', 'makbet'),
                ],
            ]
        );

        $this->add_control(
            'categories',
            [
                'label'       => esc_html__('Categories', 'makbet'),
                'type'        => Controls_Manager::SELECT2,
                'options'     => $this->get_post_categories(),
                'label_block' => true,
                'multiple'    => true,
            ]
        );

        $this->add_control(
            'cat_operator',
            [
                'label'     => esc_html__('Category Operator', 'makbet'),
                'type'      => Controls_Manager::SELECT,
                'default'   => 'IN',
                'options'   => [
                    'AND'    => esc_html__('AND', 'makbet'),
                    'IN'     => esc_html__('IN', 'makbet'),
                    'NOT IN' => esc_html__('NOT IN', 'makbet'),
                ],
                'condition' => [
                    'categories!' => ''
                ],
            ]
        );

        $this->add_control(
            'posts',
            [
                'label'       => esc_html__('Posts', 'makbet'),
                'type'        => Controls_Manager::SELECT2,
                'options'     => $this->get_posts(),
                'label_block' => true,
                'multiple'    => true,
            ]
        );

        $this->add_group_control(
            Elementor\Group_Control_Image_Size::get_type(),
            [
                'name'      => 'portfolio_image',
                'default'   => 'makbet-portfolio-large',
                'separator' => 'none',
            ]
        );

        $this->end_controls_section();

        // Wrapper.
        $this->start_controls_section(
            'section_style_wrapper',
            [
                'label' => esc_html__('Items', 'makbet'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'flex',
            [
                'label'     => esc_html__('Flex Direction', 'makbet'),
                'type'      => \Elementor\Controls_Manager::CHOOSE,
                'options'   => [
                    'row'            => [
                        'title' => esc_html__('Row', 'makbet'),
                        'icon'  => 'eicon-arrow-right',
                    ],
                    'column'         => [
                        'title' => esc_html__('Column', 'makbet'),
                        'icon'  => 'eicon-arrow-down',
                    ],
                    'row-reverse'    => [
                        'title' => esc_html__('Row Reverse', 'makbet'),
                        'icon'  => 'eicon-arrow-left',
                    ],
                    'column-reverse' => [
                        'title' => esc_html__('Column Reverse', 'makbet'),
                        'icon'  => 'eicon-arrow-up',
                    ],
                ],
                'toggle'    => true,
                'selectors' => [
                    '{{WRAPPER}} .portfolio-inner' => 'flex-direction: {{VALUE}};',
                ],
                'condition' => [
                    'portfolio-style' => ['1'],
                ],
            ]
        );

        $this->add_responsive_control(
            'wrapper_margin',
            [
                'label'      => esc_html__('Margin', 'makbet'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .portfolio-inner' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'condition'  => [
                    'portfolio-style!' => ['2'],
                ],
            ]
        );

        $this->add_control(
            'wrapper_border_radius',
            [
                'label'      => esc_html__('Border Radius', 'makbet'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .portfolio-inner' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'bg_color',
            [
                'label'     => esc_html__('Background Color', 'makbet'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .portfolio-inner' => 'background: {{VALUE}};',
                ],
                'condition' => [
                    'portfolio-style!' => ['2'],
                ],
            ]
        );

        $this->end_controls_section();


        // Image.
        $this->start_controls_section(
            'section_style_image',
            [
                'label' => esc_html__('Image', 'makbet'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'image_height',
            [
                'label'      => esc_html__('Height', 'makbet'),
                'type'       => Controls_Manager::SLIDER,
                'size_units' => ['%', 'px', 'custom'],
                'range'      => [
                    'px' => [
                        'min' => 0,
                        'max' => 1000,
                    ],
                    '%'  => [
                        'min' => 0,
                        'max' => 200,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}} .portfolio-post-thumbnail' => 'padding-top: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'image_width',
            [
                'label'      => esc_html__('Width', 'makbet'),
                'type'       => Controls_Manager::SLIDER,
                'size_units' => ['%', 'px', 'custom'],
                'range'      => [
                    'px' => [
                        'min' => 0,
                        'max' => 1000,
                    ],
                    '%'  => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}} .portfolio-post-thumbnail' => 'width: {{SIZE}}{{UNIT}};',
                ],
                'condition'  => [
                    'portfolio-style' => ['1'],
                ],
            ]
        );

        $this->add_control(
            'image_border_radius',
            [
                'label'      => esc_html__('Border Radius', 'makbet'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .portfolio-post-thumbnail' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'condition'  => [
                    'portfolio-style!' => ['2'],
                ],
            ]
        );

        $this->end_controls_section();

        //Content

        $this->start_controls_section(
            'section_style_content',
            [
                'label' => esc_html__('Content', 'makbet'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'content_padding',
            [
                'label'      => esc_html__('Padding', 'makbet'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'range'      => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}} .portfolio-content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Title.
        $this->start_controls_section(
            'section_style_title',
            [
                'label' => esc_html__('Title', 'makbet'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'title_text_color',
            [
                'label'     => esc_html__('Color', 'makbet'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .entry-title a' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .entry-title'   => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'title_text_color_hover',
            [
                'label'     => esc_html__('Hover Color', 'makbet'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .entry-title a:hover' => 'color: {{VALUE}} !important;',
                    '{{WRAPPER}} .entry-title:hover'   => 'color: {{VALUE}} !important;',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name'     => 'title_typography',
                'selector' => '{{WRAPPER}} .entry-title',
            ]
        );

        $this->add_responsive_control(
            'title_space',
            [
                'label'      => esc_html__('Margin', 'makbet'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'range'      => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}} .entry-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Category.
        $this->start_controls_section(
            'section_style_category',
            [
                'label' => esc_html__('Category', 'makbet'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'category_color',
            [
                'label'     => esc_html__('Color', 'makbet'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .portfolio-categories a' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'category_color_hover',
            [
                'label'     => esc_html__('Hover Color', 'makbet'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .portfolio-categories a:hover' => 'color: {{VALUE}} !important;',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name'     => 'category_typography',
                'selector' => '{{WRAPPER}} .portfolio-categories a',
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Border::get_type(),
            [
                'name'     => 'category_border',
                'selector' => '{{WRAPPER}} .portfolio-categories a',
            ]
        );

        $this->add_responsive_control(
            'category_border_radius',
            [
                'label'      => esc_html__('Border Radius', 'makbet'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'range'      => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}} .portfolio-categories a' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'category_padding',
            [
                'label'      => esc_html__('Padding', 'makbet'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'range'      => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}} .portfolio-categories a' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        $this->get_controls_column(['portfolio-style!' => '1']);
        $this->get_control_carousel(['portfolio-style!' => '1']);
    }

    protected function get_post_categories() {
        $categories = get_terms(array(
                'taxonomy'   => 'pavo_portfolio_cat',
                'hide_empty' => false,
            )
        );
        $results    = array();
        if (!is_wp_error($categories)) {
            foreach ($categories as $category) {
                $results[$category->slug] = $category->name;
            }
        }
        return $results;
    }

    protected function get_posts() {
        $argc = [
            'numberposts' => -1,
            'post_type'   => 'pavo_portfolio',
        ];

        $posts = get_posts($argc);

        $results = array();

        if (!empty($posts)) {
            foreach ($posts as $post) {
                $results[$post->ID] = html_entity_decode(get_the_title($post->ID), ENT_QUOTES, 'UTF-8');
            }
        }

        return $results;
    }

    protected function render() {
        $settings = $this->get_settings_for_display();

        $query = $this->query_posts();

        if (!$query->found_posts) {
            return;
        }
        $this->add_render_attribute('wrapper', 'class', [
            'elementor-portfolio-wrapper',
            'pavo_portfolio-wrapper',
            'portfolio-style-' . $settings['portfolio-style'],
        ]);
        $this->add_render_attribute('container', 'data-count', $query->post_count);
        // Item
        $this->add_render_attribute('item', 'class', [
            'elementor-portfolio-item',
            'portfolio-item-style-' . $settings['portfolio-style'],
        ]);
        $this->get_data_elementor_columns();

        ?>
        <div <?php $this->print_render_attribute_string('wrapper'); ?>>
            <div <?php $this->print_render_attribute_string('container'); ?>>
                <div <?php $this->print_render_attribute_string('inner'); ?>>
                    <?php
                    while ($query->have_posts()) {
                        $query->the_post();
                        ?>
                        <div <?php $this->print_render_attribute_string('item'); ?>>
                            <?php include get_theme_file_path('template-parts/portfolio/content-style-' . $settings['portfolio-style'] . '.php'); ?>
                        </div>
                    <?php } ?>
                </div>
            </div>
            <?php $this->get_swiper_navigation($query->post_count); ?>
        </div>
        <?php
        wp_reset_postdata();
    }

    public function query_posts() {
        $query_args = $this->get_query_args($this->get_settings());
        return new WP_Query($query_args);
    }

    public static function get_query_args($settings) {
        $query_args = [
            'post_type'           => 'pavo_portfolio',
            'orderby'             => $settings['orderby'],
            'order'               => $settings['order'],
            'ignore_sticky_posts' => 1,
            'post_status'         => 'publish', // Hide drafts/private posts for admins
        ];

        if (!empty($settings['categories'])) {
            $categories = array();
            foreach ($settings['categories'] as $category) {
                $cat = get_term_by('slug', $category, 'pavo_portfolio_cat');
                if (!is_wp_error($cat) && is_object($cat)) {
                    $categories[] = $cat->term_id;
                }
            }

            $cat_operator = isset($settings['cat_operator']) ? $settings['cat_operator'] : 'IN';

            $query_args['tax_query'] = [
                'relation' => 'AND',
                [
                    'taxonomy' => 'pavo_portfolio_cat',
                    'field'    => 'term_id',
                    'terms'    => $categories,
                    'operator' => $cat_operator,
                ],
            ];

        } elseif (!empty($settings['posts'])) {
            $query_args['post__in'] = $settings['posts'];
        }

        $query_args['posts_per_page'] = $settings['posts_per_page'];

        if (is_front_page()) {
            $query_args['paged'] = (get_query_var('page')) ? get_query_var('page') : 1;
        } else {
            $query_args['paged'] = (get_query_var('paged')) ? get_query_var('paged') : 1;
        }

        return $query_args;
    }
}

$widgets_manager->register(new Makbet_Elementor_Portfolio());