<?php

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

use Elementor\Controls_Manager;
use Elementor\Core\Kits\Documents\Tabs\Global_Colors;
use Elementor\Group_Control_Background;
use Elementor\Group_Control_Text_Shadow;
use Elementor\Group_Control_Text_Stroke;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Icons_Manager;
use Elementor\Utils;
use Elementor\Group_Control_Image_Size;
use Makbet\Elementor\Makbet_Base_Widgets;
use Elementor\Group_Control_Border;


/**
 * Elementor icon box widget.
 *
 * Elementor widget that displays an icon, a headline and a text.
 *
 * @since 1.0.0
 */
class Makbet_Widget_Icon_Box extends Makbet_Base_Widgets {

    /**
     * Get widget name.
     *
     * Retrieve icon box widget name.
     *
     * @return string Widget name.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_name() {
        return 'makbet-icon-box';
    }

    /**
     * Get widget title.
     *
     * Retrieve icon box widget title.
     *
     * @return string Widget title.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_title() {
        return __('Makbet Icon Box', 'makbet');
    }

    /**
     * Get widget icon.
     *
     * Retrieve icon box widget icon.
     *
     * @return string Widget icon.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_icon() {
        return 'eicon-icon-box';
    }

    public function get_categories() {
        return array('makbet-addons');
    }
    public function get_script_depends()
    {
        return ['makbet-elementor-icon-box'];
    }
    public function get_style_depends(): array {
        return [ 'widget-icon-box' ];
    }

    /**
     * Register icon box widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function register_controls() {
        $this->start_controls_section(
            'section_icon',
            [
                'label' => __('Icon Box', 'makbet'),
            ]
        );

        $this->add_control(
            'selected_icon',
            [
                'label'            => esc_html__('Icon', 'makbet'),
                'type'             => Controls_Manager::ICONS,
                'fa4compatibility' => 'icon',
                'default'          => [
                    'value'   => 'fas fa-star',
                    'library' => 'fa-solid',
                ],
            ]
        );
        $this->add_control(
            'view',
            [
                'label'        => esc_html__('View', 'makbet'),
                'type'         => Controls_Manager::SELECT,
                'options'      => [
                    'default' => esc_html__('Default', 'makbet'),
                    'stacked' => esc_html__('Stacked', 'makbet'),
                    'framed'  => esc_html__('Framed', 'makbet'),
                ],
                'default'      => 'default',
                'prefix_class' => 'elementor-view-',
                'condition'    => [
                    'selected_icon[value]!' => '',
                ],
            ]
        );
        $this->add_control(
            'shape',
            [
                'label'        => esc_html__('Shape', 'makbet'),
                'type'         => Controls_Manager::SELECT,
                'options'      => [
                    'circle' => esc_html__('Circle', 'makbet'),
                    'square' => esc_html__('Square', 'makbet'),
                ],
                'default'      => 'circle',
                'condition'    => [
                    'view!'                 => 'default',
                    'selected_icon[value]!' => '',
                ],
                'prefix_class' => 'elementor-shape-',
            ]
        );

        $this->add_control(
            'effect',
            [
                'label'        => esc_html__('Effect', 'makbet'),
                'type'         => Controls_Manager::SWITCHER,
                'condition'    => [
                    'selected_icon[value]!' => '',
                ],
                'prefix_class' => 'elementor-effect-',
            ]
        );
        $this->add_control(
            'title_text',
            [
                'label'       => esc_html__('Title', 'makbet'),
                'type'        => Controls_Manager::TEXT,
                'dynamic'     => [
                    'active' => true,
                ],
                'default'     => esc_html__('This is the heading', 'makbet'),
                'placeholder' => esc_html__('Enter your title', 'makbet'),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'description_text',
            [
                'label'       => esc_html__('Description Top', 'makbet'),
                'type'        => Controls_Manager::TEXTAREA,
                'dynamic'     => [
                    'active' => true,
                ],
                'default'     => esc_html__('Lorem ipsum dolor sit amet, consectetur adipiscing elit. Ut elit tellus, luctus nec ullamcorper mattis, pulvinar dapibus leo.', 'makbet'),
                'placeholder' => esc_html__('Enter your description', 'makbet'),
                'rows'        => 3,
            ]
        );

        $this->add_control(
            'description_text_2',
            [
                'label'       => esc_html__('Description Bottom', 'makbet'),
                'type'        => Controls_Manager::TEXTAREA,
                'dynamic'     => [
                    'active' => true,
                ],
                'placeholder' => esc_html__('Enter your description', 'makbet'),
                'rows'        => 3,
            ]
        );

        $this->add_control(
            'link',
            [
                'label' => esc_html__( 'Link', 'makbet' ),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'icon-box-style',
            [
                'label'        => esc_html__('Style', 'makbet'),
                'type'         => Controls_Manager::SELECT,
                'default'      => 'default',
                'options'      => [
                    'default' => esc_html__('Default', 'makbet'),
                    '1'       => esc_html__('Style 1', 'makbet'),
                    '2'       => esc_html__('Style 2', 'makbet'),
                ],
                'prefix_class' => 'icon-box-style-'
            ]
        );

        $this->add_control(
            'title_size',
            [
                'label'   => esc_html__('Title HTML Tag', 'makbet'),
                'type'    => Controls_Manager::SELECT,
                'options' => [
                    'h1'   => 'H1',
                    'h2'   => 'H2',
                    'h3'   => 'H3',
                    'h4'   => 'H4',
                    'h5'   => 'H5',
                    'h6'   => 'H6',
                    'div'  => 'div',
                    'span' => 'span',
                    'p'    => 'p',
                ],
                'default' => 'h4',
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'section_style_box',
            [
                'label' => esc_html__('Box', 'makbet'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'icon_box_background_color',
            [
                'label'     => __('Background Color', 'makbet'),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .elementor-icon-box-wrapper' => 'background-color: {{VALUE}};',
                ],
            ]
        );
        $this->add_control(
            'background_color_gradient',
            [
                'label'        => esc_html__('Background Gradient Effect', 'makbet'),
                'type'         => Controls_Manager::SWITCHER,
                'prefix_class' => 'elementor-background-gradient-',
            ]
        );
        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name'  => 'background_overlay_color',
                'types' => [ 'gradient' ],
                'fields_options' => [
                    'background' => [
                        'label'               => __( 'Background Gradient Hover', 'makbet' ),
                        'frontend_available'  => true,
                    ],
                ],
                'selector'  => '{{WRAPPER}}.elementor-background-gradient-yes::before',
                'condition' => [
                    'background_color_gradient' => 'yes',
                ],
            ]
        );
        $this->add_control(
            'icon_box_background_color_hover',
            [
                'label'     => __('Background Color Hover', 'makbet'),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}}:hover' => 'background-color: {{VALUE}};',
                ],
                'condition' => [
                    'background_color_gradient!' => 'yes',
                ],
            ]
        );
        $this->add_responsive_control(
            'position',
            [
                'label'          => esc_html__('Icon Position', 'makbet'),
                'type'           => Controls_Manager::CHOOSE,
                'default'        => 'top',
                'mobile_default' => 'top',
                'options'        => [
                    'left'  => [
                        'title' => esc_html__('Left', 'makbet'),
                        'icon'  => 'eicon-h-align-left',
                    ],
                    'top'   => [
                        'title' => esc_html__('Top', 'makbet'),
                        'icon'  => 'eicon-v-align-top',
                    ],
                    'right' => [
                        'title' => esc_html__('Right', 'makbet'),
                        'icon'  => 'eicon-h-align-right',
                    ],
                ],
                'prefix_class'   => 'elementor%s-position-',
                'condition'      => [
                    'selected_icon[value]!' => '',
                ],
                'separator'   => 'before',
            ]
        );

        $this->add_control(
            'content_vertical_alignment',
            [
                'label'        => esc_html__('Vertical Alignment', 'makbet'),
                'type'         => Controls_Manager::CHOOSE,
                'options'      => [
                    'top'    => [
                        'title' => esc_html__('Top', 'makbet'),
                        'icon'  => 'eicon-v-align-top',
                    ],
                    'middle' => [
                        'title' => esc_html__('Middle', 'makbet'),
                        'icon'  => 'eicon-v-align-middle',
                    ],
                    'bottom' => [
                        'title' => esc_html__('Bottom', 'makbet'),
                        'icon'  => 'eicon-v-align-bottom',
                    ],
                ],
                'default'      => 'top',
                'toggle'       => false,
                'prefix_class' => 'elementor-vertical-align-',
                'condition'    => [
                    'position!' => 'top',
                ],
            ]
        );

        $this->add_responsive_control(
            'text_align',
            [
                'label'     => esc_html__('Alignment', 'makbet'),
                'type'      => Controls_Manager::CHOOSE,
                'options'   => [
                    'left'    => [
                        'title' => esc_html__('Left', 'makbet'),
                        'icon'  => 'eicon-text-align-left',
                    ],
                    'center'  => [
                        'title' => esc_html__('Center', 'makbet'),
                        'icon'  => 'eicon-text-align-center',
                    ],
                    'right'   => [
                        'title' => esc_html__('Right', 'makbet'),
                        'icon'  => 'eicon-text-align-right',
                    ],
                    'justify' => [
                        'title' => esc_html__('Justified', 'makbet'),
                        'icon'  => 'eicon-text-align-justify',
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .elementor-icon-box-wrapper' => 'text-align: {{VALUE}};',
                ],
            ]
        );
        $this->add_responsive_control(
            'box_padding',
            [
                'label'      => esc_html__('Padding', 'makbet'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .elementor-icon-box-wrapper' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name'        => 'box_border',
                'placeholder' => '1px',
                'default'     => '1px',
                'selector'    => '{{WRAPPER}} .elementor-icon-box-wrapper',
                'separator'   => 'before',
            ]
        );
        $this->add_control(
            'box_border_radius_icon_box',
            [
                'label'      => esc_html__('Border Radius', 'makbet'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .elementor-icon-box-wrapper' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}}.elementor-background-gradient-yes::before' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name'     => 'icon-box-box-shadow',
                'selector' => '{{WRAPPER}} .elementor-icon-box-wrapper',
            ]
        );
        $this->add_control(
            'backdrop_blur_icon_box',
            [
                'label' => esc_html__('Backdrop Blur', 'makbet'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .elementor-icon-box-wrapper' => 'backdrop-filter: blur({{SIZE}}px); -webkit-backdrop-filter: blur({{SIZE}}px);',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'section_style_icon',
            [
                'label'     => esc_html__('Icon', 'makbet'),
                'tab'       => Controls_Manager::TAB_STYLE,
                'condition' => [
                    'selected_icon[value]!' => '',
                ],
            ]
        );

        $this->start_controls_tabs('icon_colors');

        $this->start_controls_tab(
            'icon_colors_normal',
            [
                'label'     => esc_html__('Normal', 'makbet'),
            ]
        );
        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name'     => 'icon_gradient_color',
                'types'    => ['gradient'],
                'fields_options' => [
                    'background' => [
                        'label'              => __( 'Gradient Icon', 'makbet' ),
                        'frontend_available' => true,
                    ],
                ],
                'selector' => '{{WRAPPER}}.elementor-view-framed .elementor-icon i:before',
                'condition' => [
                    'view' => 'framed',
                ],
            ]
        );
        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name'     => 'background_color_overlay',
                'types'    => ['gradient'],
                'fields_options' => [
                    'background' => [
                        'label'              => __( 'Background Gradient', 'makbet' ),
                        'frontend_available' => true,
                    ],
                ],
                'selector' => '{{WRAPPER}}.elementor-view-framed .elementor-icon',
                'condition' => [
                    'view' => 'framed',
                ],
            ]

        );
        $this->add_responsive_control(
            'border_width',
            [
                'label'      => esc_html__('Border Width', 'makbet'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em', 'rem', 'vw', 'custom'],
                'selectors'  => [
                    '{{WRAPPER}} .elementor-icon' => 'border-width: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'condition'  => [
                    'view' => 'framed',
                ],
            ]
        );
        $this->add_control(
            'icon_border_color',
            [
                'label'     => esc_html__('Border Color', 'makbet'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .elementor-icon-box-icon .elementor-icon' => 'border-color: {{VALUE}} !important',
                ],
                'condition' => [
                    'view' => 'framed',
                ],
            ]
        );
        $this->add_control(
            'color_gradient',
            [
                'label'        => esc_html__('Icon Gradient', 'makbet'),
                'type'         => Controls_Manager::SWITCHER,
                'prefix_class' => 'elementor-gradient-',
            ]
        );
        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name'  => 'overlay_color_icon',
                'types' => [ 'gradient' ],
                'fields_options' => [
                    'background' => [
                        'label'               => __( 'Gradient Icon', 'makbet' ),
                        'frontend_available'  => true,
                    ],
                ],
                'selector'  => '{{WRAPPER}}.elementor-gradient-yes .elementor-icon i:before',
                'condition' => [
                    'color_gradient' => 'yes',
                ],
            ]
        );
        $this->add_control(
            'primary_color',
            [
                'label'     => esc_html__('Primary Color', 'makbet'),
                'type'      => Controls_Manager::COLOR,
                'global'    => [
                    'default' => Global_Colors::COLOR_PRIMARY,
                ],
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}}.elementor-view-stacked .elementor-icon'                                                    => 'background-color: {{VALUE}};',
                    '{{WRAPPER}}.elementor-view-framed .elementor-icon, {{WRAPPER}}.elementor-view-default .elementor-icon' => 'fill: {{VALUE}}; color: {{VALUE}}; border-color: {{VALUE}};',
                ],
                'condition' => [
                    'view!' => 'framed',
                ],
            ]
        );

        $this->add_control(
            'secondary_color',
            [
                'label'     => esc_html__('Secondary Color', 'makbet'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'condition' => [
                    'view!' => ['default', 'framed' ],
                    'color_gradient!' => 'yes'
                ],

                'selectors' => [
                    '{{WRAPPER}}.elementor-view-framed .elementor-icon'                                => 'background-color: {{VALUE}};',
                    '{{WRAPPER}}.elementor-view-stacked .elementor-icon'                               => 'fill: {{VALUE}}; color: {{VALUE}};',
                    '{{WRAPPER}}.elementor-effect-yes .elementor-icon-box-icon .elementor-icon:before' => 'border-top-color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'icon_colors_hover',
            [
                'label'     => esc_html__('Hover', 'makbet'),

            ]
        );
        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name'     => 'icon_gradient_color_hover',
                'types'    => ['gradient'],
                'fields_options' => [
                    'background' => [
                        'label'              => __( 'Gradient Icon Hover', 'makbet' ),
                        'frontend_available' => true,
                    ],
                ],
                'selector' => '{{WRAPPER}}.elementor-view-framed:hover .elementor-icon i:before',
                'condition' => [
                    'view' => 'framed',
                ],
            ]
        );
        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name'     => 'background_color_overlay_hover',
                'types'    => ['gradient'],
                'fields_options' => [
                    'background' => [
                        'label'              => __( 'Background Gradient Hover', 'makbet' ),
                        'frontend_available' => true,
                    ],
                ],
                'selector' => '{{WRAPPER}}.elementor-view-framed:hover .elementor-icon',
                'condition' => [
                    'view' => 'framed',
                ],
            ]

        );
        $this->add_responsive_control(
            'border_width_hover',
            [
                'label'      => esc_html__('Border Width', 'makbet'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em', 'rem', 'vw', 'custom'],
                'selectors'  => [
                    '{{WRAPPER}}:hover .elementor-icon' => 'border-width: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'condition'  => [
                    'view' => 'framed',
                ],
            ]
        );
        $this->add_control(
            'icon_border_color_hover',
            [
                'label'     => esc_html__('Border Color', 'makbet'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}}:hover .elementor-icon-box-icon .elementor-icon' => 'border-color: {{VALUE}} !important',
                ],
                'condition' => [
                    'view' => 'framed',
                ],
            ]
        );
        $this->add_control(
            'hover_primary_color',
            [
                'label'     => esc_html__('Primary Color', 'makbet'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}}.elementor-view-stacked:hover .elementor-icon' => 'background-color: {{VALUE}};',
                    '{{WRAPPER}}.elementor-view-framed .elementor-icon:hover'  => 'fill: {{VALUE}}; color: {{VALUE}}; border-color: {{VALUE}};',
                    '{{WRAPPER}}:hover .elementor-icon'                        => 'fill: {{VALUE}}; color: {{VALUE}};',
                ],
                'condition' => [
                    'view!' => 'framed',
                ],
            ]
        );

        $this->add_control(
            'hover_secondary_color',
            [
                'label'     => esc_html__('Secondary Color', 'makbet'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'condition' => [
                    'view!' => ['default', 'framed'],
                    'color_gradient!' => 'yes'
                ],
                'selectors' => [
                    '{{WRAPPER}}.elementor-view-framed .elementor-icon:hover'  => 'background-color: {{VALUE}};',
                    '{{WRAPPER}}.elementor-view-stacked:hover .elementor-icon' => 'fill: {{VALUE}}; color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'hover_animation',
            [
                'label'     => esc_html__('Hover Animation', 'makbet'),
                'type'      => Controls_Manager::HOVER_ANIMATION,
                'condition' => [
                    'view!' => 'framed',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();


        $this->add_responsive_control(
            'icon_space',
            [
                'label'      => esc_html__('Spacing Icon', 'makbet'),
                'type'       => Controls_Manager::SLIDER,
                'size_units' => ['px', '%', 'em', 'rem', 'vw', 'custom'],
                'default'    => [
                    'size' => 30,
                ],
                'range'      => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}}' => '--icon-box-icon-margin: {{SIZE}}{{UNIT}}',
                ],
            ]
        );

        $this->add_responsive_control(
            'icon_size',
            [
                'label'      => esc_html__('Size', 'makbet'),
                'type'       => Controls_Manager::SLIDER,
                'size_units' => ['px', '%', 'em', 'rem', 'vw', 'custom'],
                'range'      => [
                    'px' => [
                        'min' => 6,
                        'max' => 300,
                    ],
                ],
                'selectors'  => [
                    '{{WRAPPER}} .elementor-icon' => 'font-size: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        $this->add_responsive_control(
            'ic_width_svg',
            [
                'label'          => esc_html__('Width SVG', 'makbet'),
                'type'           => Controls_Manager::SLIDER,
                'default'        => [
                    'unit' => 'px',
                ],
                'tablet_default' => [
                    'unit' => 'px',
                ],
                'mobile_default' => [
                    'unit' => 'px',
                ],
                'size_units'     => ['%', 'px', 'vw'],
                'range'          => [
                    '%'  => [
                        'min' => 1,
                        'max' => 100,
                    ],
                    'px' => [
                        'min' => 1,
                        'max' => 1000,
                    ],
                    'vw' => [
                        'min' => 1,
                        'max' => 100,
                    ],
                ],
                'selectors'      => [
                    '{{WRAPPER}} .elementor-icon-box-icon .elementor-icon .icon-box-mask' => 'width: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        $this->add_responsive_control(
            'ic_height_svg',
            [
                'label'          => esc_html__('Height SVG', 'makbet'),
                'type'           => Controls_Manager::SLIDER,
                'default'        => [
                    'unit' => 'px',
                ],
                'tablet_default' => [
                    'unit' => 'px',
                ],
                'mobile_default' => [
                    'unit' => 'px',
                ],
                'size_units'     => ['%', 'px', 'vw'],
                'range'          => [
                    '%'  => [
                        'min' => 1,
                        'max' => 100,
                    ],
                    'px' => [
                        'min' => 1,
                        'max' => 1000,
                    ],
                    'vw' => [
                        'min' => 1,
                        'max' => 100,
                    ],
                ],
                'selectors'      => [
                    '{{WRAPPER}} .elementor-icon-box-icon .elementor-icon .icon-box-mask' => 'height: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        $this->add_responsive_control(
            'align_icon',
            [
                'label'     => esc_html__('Alignment Info ', 'makbet'),
                'type'      => Controls_Manager::CHOOSE,
                'options'   => [
                    'flex-start' => [
                        'title' => esc_html__('Left', 'makbet'),
                        'icon'  => 'eicon-h-align-left',
                    ],
                    'center'     => [
                        'title' => esc_html__('Center', 'makbet'),
                        'icon'  => 'eicon-h-align-center',
                    ],
                    'flex-end'   => [
                        'title' => esc_html__('Right', 'makbet'),
                        'icon'  => 'eicon-h-align-right',
                    ],
                ],
                'default'   => 'center',
                'selectors' => [
                    '{{WRAPPER}} .elementor-icon-box-icon'                 => 'justify-content: {{VALUE}}',
                    '{{WRAPPER}} .elementor-icon-box-icon .elementor-icon' => 'justify-content: {{VALUE}}',
                ],
            ]
        );
        $this->add_responsive_control(
            'padding_icon_box',
            [
                'label'      => esc_html__('Padding', 'makbet'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .elementor-icon-box-icon .elementor-icon' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        $this->add_responsive_control(
            'icon_box_margin',
            [
                'label'      => esc_html__('Margin', 'makbet'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .elementor-icon-box-icon .elementor-icon i'   => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .elementor-icon-box-icon .elementor-icon .icon-box-mask' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );


        $this->add_responsive_control(
            'border_radius',
            [
                'label'      => esc_html__('Border Radius', 'makbet'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em', 'rem', 'custom'],
                'selectors'  => [
                    '{{WRAPPER}} .elementor-icon' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'condition'  => [
                    'view!' => 'default',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'section_style_content',
            [
                'label' => esc_html__('Content', 'makbet'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'icon_box_content_background',
            [
                'label'     => esc_html__('Background Color', 'makbet'),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .elementor-icon-box-content'   => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'icon_box_content_padding',
            [
                'label'      => esc_html__('Content Padding', 'makbet'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em'],
                'selectors'  => [
                    '{{WRAPPER}} .elementor-icon-box-content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
                ],
            ]
        );

        $this->add_responsive_control(
            'icon_box_content_margin',
            [
                'label'      => esc_html__('Content Margin', 'makbet'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em'],
                'selectors'  => [
                    '{{WRAPPER}} .elementor-icon-box-content' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
                ],
            ]
        );

        $this->add_responsive_control(
            'icon_box_content_radius',
            [
                'label'      => esc_html__('Border Radius', 'makbet'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em'],
                'selectors'  => [
                    '{{WRAPPER}} .elementor-icon-box-content' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
                ],
            ]
        );

        $this->add_control(
            'heading_title',
            [
                'label'     => esc_html__('Title', 'makbet'),
                'type'      => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'effect_heading_title',
            [
                'label'        => esc_html__('Effect Title', 'makbet'),
                'type'         => Controls_Manager::SWITCHER,
                'condition'    => [
                    'selected_icon[value]!' => '',
                    'shape!'                => 'square',
                ],
                'prefix_class' => 'title-effect-',
            ]
        );
        $this->add_responsive_control(
            'icon_box_title_margin',
            [
                'label'      => esc_html__('Title Margin', 'makbet'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em'],
                'selectors'  => [
                    '{{WRAPPER}} .elementor-icon-box-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
                ],
            ]
        );

        $this->add_control(
            'title_color',
            [
                'label'     => esc_html__('Color', 'makbet'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .elementor-icon-box-title'   => 'color: {{VALUE}};',
                    '{{WRAPPER}} .elementor-icon-box-title a' => 'color: {{VALUE}};',
                ],
                'global'    => [
                    'default' => Global_Colors::COLOR_PRIMARY,
                ],
            ]
        );
        $this->add_control(
            'icon_box_title_hover',
            [
                'label'     => esc_html__('Color Title Hover', 'makbet'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}}:hover .elementor-icon-box-content .elementor-icon-box-title'   => 'color: {{VALUE}};',
                    '{{WRAPPER}}:hover .elementor-icon-box-content .elementor-icon-box-title a' => 'color: {{VALUE}};',

                ],
            ]
        );
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name'     => 'title_typography',
                'selector' => '{{WRAPPER}} .elementor-icon-box-title, {{WRAPPER}} .elementor-icon-box-title a',
            ]
        );

        $this->add_group_control(
            Group_Control_Text_Stroke::get_type(),
            [
                'name'     => 'text_stroke',
                'selector' => '{{WRAPPER}} .elementor-icon-box-title',
            ]
        );

        $this->add_group_control(
            Group_Control_Text_Shadow::get_type(),
            [
                'name'     => 'title_shadow',
                'selector' => '{{WRAPPER}} .elementor-icon-box-title',
            ]
        );

        $this->add_control(
            'heading_description',
            [
                'label'     => esc_html__('Description Top', 'makbet'),
                'type'      => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'description_color',
            [
                'label'     => esc_html__('Color', 'makbet'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .elementor-icon-box-description' => 'color: {{VALUE}};',
                ],
                'global'    => [
                    'default' => Global_Colors::COLOR_TEXT,
                ],
            ]
        );
        $this->add_control(
            'icon_box_description_hover',
            [
                'label'     => esc_html__('Color Description Hover', 'makbet'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}}:hover .elementor-icon-box-content .elementor-icon-box-description' => 'color: {{VALUE}};',
                ],
            ]
        );
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name'     => 'description_typography',
                'selector' => '{{WRAPPER}} .elementor-icon-box-description',
            ]
        );

        $this->add_group_control(
            Group_Control_Text_Shadow::get_type(),
            [
                'name'     => 'description_shadow',
                'selector' => '{{WRAPPER}} .elementor-icon-box-description',
            ]
        );

        $this->add_responsive_control(
            'icon_box_description_padding',
            [
                'label'      => esc_html__('Description Top Padding', 'makbet'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em'],
                'selectors'  => [
                    '{{WRAPPER}} .elementor-icon-box-description' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
                ],
            ]
        );

        $this->add_control(
            'heading_description_2',
            [
                'label'     => esc_html__('Description Bottom', 'makbet'),
                'type'      => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'description_color_2',
            [
                'label'     => esc_html__('Color', 'makbet'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .elementor-icon-box-description-2' => 'color: {{VALUE}};',
                ],
                'global'    => [
                    'default' => Global_Colors::COLOR_TEXT,
                ],
            ]
        );
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name'     => 'description_typography_2',
                'selector' => '{{WRAPPER}} .elementor-icon-box-description-2',
            ]
        );

        $this->end_controls_section();

    }

    protected function render() {
        $settings = $this->get_settings_for_display();

        $this->add_render_attribute( 'icon', 'class', [ 'elementor-icon', 'elementor-animation-' . $settings['hover_animation'] ] );
        $has_link = ! empty( $settings['link']['url'] );
        $icon_tag = $has_link ? 'a' : 'span';
        $button_tag = 'span';

        if ( ! isset( $settings['icon'] ) && ! Icons_Manager::is_migration_allowed() ) {
            // add old default
            $settings['icon'] = 'fa fa-star';
        }

        $has_icon = ! empty( $settings['icon'] );

        if ( ! empty( $settings['link']['url'] ) ) {
            $button_tag = 'a';
            $this->add_link_attributes( 'link', $settings['link'] );
        }

        if ( $has_icon ) {
            $this->add_render_attribute( 'i', 'class', $settings['icon'] );
            $this->add_render_attribute( 'i', 'aria-hidden', 'true' );
        }

        $this->add_render_attribute( 'description_text', 'class', 'elementor-icon-box-description' );
        $this->add_render_attribute( 'description_text_2', 'class', 'elementor-icon-box-description-2' );

        $this->add_inline_editing_attributes( 'title_text', 'none' );
        $this->add_inline_editing_attributes( 'description_text' );
        $this->add_inline_editing_attributes( 'description_text_2' );
        if ( ! $has_icon && ! empty( $settings['selected_icon']['value'] ) ) {
            $has_icon = true;
        }
        $migrated = isset( $settings['__fa4_migrated']['selected_icon'] );
        $is_new = ! isset( $settings['icon'] ) && Icons_Manager::is_migration_allowed();

        ?>
        <div class="elementor-icon-box-wrapper">
            <?php if ( $has_icon ) : ?>
                <div class="elementor-icon-box-icon">
                    <<?php Utils::print_validated_html_tag( $icon_tag ); ?> <?php $this->print_render_attribute_string( 'icon' ); ?> <?php $this->print_render_attribute_string( 'link' ); ?>>
                        <?php
                        $icon = $settings['selected_icon'] ?? $settings['icon'] ?? null;
                        if ( is_array($icon) && !empty($icon['library']) ) {
                            if ( $icon['library'] === 'svg' ) {
                                $svg_url = '';
                                if ( !empty( $icon['value']['url'] ) ) {
                                    $svg_url = $icon['value']['url'];
                                } elseif ( is_string( $icon['value'] ) && filter_var( $icon['value'], FILTER_VALIDATE_URL ) ) {
                                    $svg_url = $icon['value'];
                                }

                                if ( $svg_url ) {
                                    echo '<span class="icon-box-mask" style="--icon-url: url(\'' . esc_url( $svg_url ) . '\')"></span>';
                                }
                            } else {
                                \Elementor\Icons_Manager::render_icon( $icon, [ 'aria-hidden' => 'true', 'class' => 'icon-box-icon' ] );
                            }
                        }
                        ?>

                    </<?php Utils::print_validated_html_tag( $icon_tag ); ?>>
                </div>
            <?php endif; ?>

            <div class="elementor-icon-box-content">
        <div class="title-content">
        <?php if (!Utils::is_empty($settings['title_text'])) : ?>
        <<?php Utils::print_validated_html_tag( $settings['title_size'] ); ?> class="elementor-icon-box-title">
        <<?php Utils::print_validated_html_tag( $icon_tag ); ?>  <?php $this->print_render_attribute_string( 'title_text' ); ?> <?php $this->print_render_attribute_string( 'link' ); ?>>
        <?php $this->print_unescaped_setting( 'title_text' ); ?>
        </<?php Utils::print_validated_html_tag( $icon_tag ); ?>>
        </<?php Utils::print_validated_html_tag( $settings['title_size'] ); ?>>
        <?php endif; ?>
        <?php if ( ! Utils::is_empty( $settings['description_text'] ) ) : ?>
            <p <?php $this->print_render_attribute_string( 'description_text' ); ?>>
                <?php $this->print_unescaped_setting( 'description_text' ); ?>
            </p>
        <?php endif; ?>

        <?php if ( ! Utils::is_empty( $settings['description_text_2'] ) ) : ?>
            <p <?php $this->print_render_attribute_string( 'description_text_2' ); ?>>
                <?php $this->print_unescaped_setting( 'description_text_2' ); ?>
            </p>
        <?php endif; ?>
        </div>
        </div>
        </div>
        <?php
    }
}

$widgets_manager->register(new Makbet_Widget_Icon_Box());